'use strict';

redirectrules.controllers.module('redirectrules-overview', ['xhr'], function () {
	return {
		vue: true,
		name: 'gx-redirectrules-overview',
		template: '#redirectrules-overview',
		props: ['base_url'],
		data: function data() {
			return {
				configuration: {
					maxAge: 3600
				},
				expertMode: false,
				allRules: [],
				//rules: [],
				add: {
					urlPath: '',
					queryMatchMode: 'match',
					query: '',
					responseCode: 301,
					target: '',
					queryProcessing: 'drop'
				},
				edit: {
					id: 0,
					urlPath: '',
					queryMatchMode: 'match',
					query: '',
					responseCode: 301,
					target: '',
					queryProcessing: 'drop',
					status: true,
					maxAge: 3600
				},
				pagination: {
					page: 1,
					elementsPerPage: 10,
					limits: [10, 25, 50, 100, 200]
				},
				lang: jse.core.lang.getSection('redirectrules')
			};
		},
		computed: {
			maxPages: function maxPages() {
				return Math.ceil(this.allRules.length / this.pagination.elementsPerPage);
			},
			currentPageStart: function currentPageStart() {
				return (this.pagination.page - 1) * this.pagination.elementsPerPage;
			},
			currentPageEnd: function currentPageEnd() {
				return this.currentPageStart + this.pagination.elementsPerPage;
			},
			totalElementsCount: function totalElementsCount() {
				return this.allRules.length;
			},
			rules: function rules() {
				return this.allRules.slice(this.currentPageStart, this.currentPageEnd);
			}
		},
		watch: {},
		methods: {
			loadConfiguration: function loadConfiguration() {
				var component = this;
				jse.libs.xhr.get({
					url: 'redirect-rules/get-configuration'
				}).done(function (response) {
					component.configuration = response;
				});
			},
			saveConfiguration: function saveConfiguration() {
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/save-configuration',
					data: component.configuration
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					} else {
						window.console.info('configuration saved');
					}
				});
			},
			loadRedirectRules: function loadRedirectRules() {
				var component = this;
				return jse.libs.xhr.get({
					url: 'redirect-rules/get-rules'
				}).then(function (response) {
					component.allRules = response.rules;
				});
			},
			addRule: function addRule() {
				if (!this.expertMode) {
					this.add.queryMatchMode = 'match';
					this.add.queryProcessing = 'drop';
					this.add.responseCode = 301;
				}
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/add-rule',
					data: {
						newRule: component.add
					}
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					} else {
						component.add.urlPath = '';
						component.add.queryMatchMode = 'ignore';
						component.add.query = '';
						component.add.responseCode = 302;
						component.add.target = '';
						component.add.queryProcessing = 'merge';
						component.loadRedirectRules().then(function () {
							component.pagination.page = component.maxPages;
						});
					}
				});
			},
			deleteRule: function deleteRule(ruleId) {
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/delete-rule',
					data: {
						ruleId: ruleId
					}
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					}
					component.loadRedirectRules().then(function () {
						if (component.pagination.page > component.maxPages) {
							component.pagination.page = component.maxPages;
						}
					});
				});
			},
			changeStatus: function changeStatus(newStatus, ruleId) {
				if (newStatus === true) {
					this.enableRule(ruleId);
				} else {
					this.disableRule(ruleId);
				}
			},
			enableRule: function enableRule(ruleId) {
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/enable-rule',
					data: {
						ruleId: ruleId
					}
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					}
					component.loadRedirectRules();
				});
			},
			disableRule: function disableRule(ruleId) {
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/disable-rule',
					data: {
						ruleId: ruleId
					}
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					}
					component.loadRedirectRules();
				});
			},
			editRule: function editRule(event, ruleindex) {
				var component = this;
				var row = event.target.closest('tr');
				var table = event.target.closest('table');
				table.querySelectorAll('tr.edit').forEach(function (el) {
					el.classList.remove('edit');
				});
				this.edit = this.rules[ruleindex];
				if (this.edit.query) {
					this.edit.urlPath += '?' + this.edit.query;
					this.edit.query = '';
				}
				row.classList.add('edit');
			},
			saveRule: function saveRule(event) {
				var component = this;
				jse.libs.xhr.post({
					url: 'redirect-rules/update-rule',
					data: {
						ruleData: component.edit
					}
				}).then(function (response) {
					if (response.message) {
						window.alert(jse.core.lang.translate(response.message, 'redirectrules'));
					} else {
						var table = event.target.closest('table');
						table.querySelectorAll('tr.edit').forEach(function (el) {
							el.classList.remove('edit');
						});
						component.loadRedirectRules();
					}
				});
			}
		},
		mounted: function mounted() {
			this.loadConfiguration();
			this.loadRedirectRules();
		},

		init: function init(done) {
			return done();
		}
	};
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlZGlyZWN0cnVsZXMvcmVkaXJlY3RydWxlcy1vdmVydmlldy5qcyJdLCJuYW1lcyI6WyJyZWRpcmVjdHJ1bGVzIiwiY29udHJvbGxlcnMiLCJtb2R1bGUiLCJ2dWUiLCJuYW1lIiwidGVtcGxhdGUiLCJwcm9wcyIsImRhdGEiLCJjb25maWd1cmF0aW9uIiwibWF4QWdlIiwiZXhwZXJ0TW9kZSIsImFsbFJ1bGVzIiwiYWRkIiwidXJsUGF0aCIsInF1ZXJ5TWF0Y2hNb2RlIiwicXVlcnkiLCJyZXNwb25zZUNvZGUiLCJ0YXJnZXQiLCJxdWVyeVByb2Nlc3NpbmciLCJlZGl0IiwiaWQiLCJzdGF0dXMiLCJwYWdpbmF0aW9uIiwicGFnZSIsImVsZW1lbnRzUGVyUGFnZSIsImxpbWl0cyIsImxhbmciLCJqc2UiLCJjb3JlIiwiZ2V0U2VjdGlvbiIsImNvbXB1dGVkIiwibWF4UGFnZXMiLCJNYXRoIiwiY2VpbCIsImxlbmd0aCIsImN1cnJlbnRQYWdlU3RhcnQiLCJjdXJyZW50UGFnZUVuZCIsInRvdGFsRWxlbWVudHNDb3VudCIsInJ1bGVzIiwic2xpY2UiLCJ3YXRjaCIsIm1ldGhvZHMiLCJsb2FkQ29uZmlndXJhdGlvbiIsImNvbXBvbmVudCIsImxpYnMiLCJ4aHIiLCJnZXQiLCJ1cmwiLCJkb25lIiwicmVzcG9uc2UiLCJzYXZlQ29uZmlndXJhdGlvbiIsInBvc3QiLCJ0aGVuIiwibWVzc2FnZSIsIndpbmRvdyIsImFsZXJ0IiwidHJhbnNsYXRlIiwiY29uc29sZSIsImluZm8iLCJsb2FkUmVkaXJlY3RSdWxlcyIsImFkZFJ1bGUiLCJuZXdSdWxlIiwiZGVsZXRlUnVsZSIsInJ1bGVJZCIsImNoYW5nZVN0YXR1cyIsIm5ld1N0YXR1cyIsImVuYWJsZVJ1bGUiLCJkaXNhYmxlUnVsZSIsImVkaXRSdWxlIiwiZXZlbnQiLCJydWxlaW5kZXgiLCJyb3ciLCJjbG9zZXN0IiwidGFibGUiLCJxdWVyeVNlbGVjdG9yQWxsIiwiZm9yRWFjaCIsImVsIiwiY2xhc3NMaXN0IiwicmVtb3ZlIiwic2F2ZVJ1bGUiLCJydWxlRGF0YSIsIm1vdW50ZWQiLCJpbml0Il0sIm1hcHBpbmdzIjoiOztBQUFBQSxjQUFjQyxXQUFkLENBQTBCQyxNQUExQixDQUNDLHdCQURELEVBRUMsQ0FBQyxLQUFELENBRkQsRUFHQyxZQUFNO0FBQ0wsUUFBTztBQUNOQyxPQUFLLElBREM7QUFFTkMsUUFBTSwyQkFGQTtBQUdOQyxZQUFVLHlCQUhKO0FBSU5DLFNBQU8sQ0FBQyxVQUFELENBSkQ7QUFLTkMsUUFBTTtBQUFBLFVBQU87QUFDWkMsbUJBQWU7QUFDZEMsYUFBUTtBQURNLEtBREg7QUFJWkMsZ0JBQVksS0FKQTtBQUtaQyxjQUFVLEVBTEU7QUFNWjtBQUNBQyxTQUFLO0FBQ0pDLGNBQVMsRUFETDtBQUVKQyxxQkFBZ0IsT0FGWjtBQUdKQyxZQUFPLEVBSEg7QUFJSkMsbUJBQWMsR0FKVjtBQUtKQyxhQUFRLEVBTEo7QUFNSkMsc0JBQWlCO0FBTmIsS0FQTztBQWVaQyxVQUFNO0FBQ0xDLFNBQUksQ0FEQztBQUVMUCxjQUFTLEVBRko7QUFHTEMscUJBQWdCLE9BSFg7QUFJTEMsWUFBTyxFQUpGO0FBS0xDLG1CQUFjLEdBTFQ7QUFNTEMsYUFBUSxFQU5IO0FBT0xDLHNCQUFpQixNQVBaO0FBUUxHLGFBQVEsSUFSSDtBQVNMWixhQUFRO0FBVEgsS0FmTTtBQTBCWmEsZ0JBQVk7QUFDWEMsV0FBTSxDQURLO0FBRVhDLHNCQUFpQixFQUZOO0FBR1hDLGFBQVEsQ0FBQyxFQUFELEVBQUssRUFBTCxFQUFTLEVBQVQsRUFBYSxHQUFiLEVBQWtCLEdBQWxCO0FBSEcsS0ExQkE7QUErQlpDLFVBQU1DLElBQUlDLElBQUosQ0FBU0YsSUFBVCxDQUFjRyxVQUFkLENBQXlCLGVBQXpCO0FBL0JNLElBQVA7QUFBQSxHQUxBO0FBc0NOQyxZQUFVO0FBQ1RDLFdBRFMsc0JBQ0U7QUFDVixXQUFPQyxLQUFLQyxJQUFMLENBQVUsS0FBS3RCLFFBQUwsQ0FBY3VCLE1BQWQsR0FBdUIsS0FBS1osVUFBTCxDQUFnQkUsZUFBakQsQ0FBUDtBQUNBLElBSFE7QUFJVFcsbUJBSlMsOEJBSVU7QUFDbEIsV0FBTyxDQUFDLEtBQUtiLFVBQUwsQ0FBZ0JDLElBQWhCLEdBQXVCLENBQXhCLElBQTZCLEtBQUtELFVBQUwsQ0FBZ0JFLGVBQXBEO0FBQ0EsSUFOUTtBQU9UWSxpQkFQUyw0QkFPUTtBQUNoQixXQUFPLEtBQUtELGdCQUFMLEdBQXdCLEtBQUtiLFVBQUwsQ0FBZ0JFLGVBQS9DO0FBQ0EsSUFUUTtBQVVUYSxxQkFWUyxnQ0FVWTtBQUNwQixXQUFPLEtBQUsxQixRQUFMLENBQWN1QixNQUFyQjtBQUNBLElBWlE7QUFhVEksUUFiUyxtQkFhRDtBQUNQLFdBQU8sS0FBSzNCLFFBQUwsQ0FBYzRCLEtBQWQsQ0FBb0IsS0FBS0osZ0JBQXpCLEVBQTJDLEtBQUtDLGNBQWhELENBQVA7QUFDQTtBQWZRLEdBdENKO0FBdUROSSxTQUFPLEVBdkREO0FBd0ROQyxXQUFTO0FBQ1JDLG9CQURRLCtCQUNZO0FBQ25CLFFBQUlDLFlBQVksSUFBaEI7QUFDQWhCLFFBQUlpQixJQUFKLENBQVNDLEdBQVQsQ0FBYUMsR0FBYixDQUFpQjtBQUNoQkMsVUFBSztBQURXLEtBQWpCLEVBRUdDLElBRkgsQ0FFUSxvQkFBWTtBQUNuQkwsZUFBVW5DLGFBQVYsR0FBMEJ5QyxRQUExQjtBQUNBLEtBSkQ7QUFLQSxJQVJPO0FBU1JDLG9CQVRRLCtCQVNZO0FBQ25CLFFBQUlQLFlBQVksSUFBaEI7QUFDQWhCLFFBQUlpQixJQUFKLENBQVNDLEdBQVQsQ0FBYU0sSUFBYixDQUFrQjtBQUNqQkosVUFBSyxtQ0FEWTtBQUVqQnhDLFdBQU1vQyxVQUFVbkM7QUFGQyxLQUFsQixFQUdHNEMsSUFISCxDQUdRLFVBQUNILFFBQUQsRUFBYztBQUNyQixTQUFJQSxTQUFTSSxPQUFiLEVBQXNCO0FBQ3JCQyxhQUFPQyxLQUFQLENBQWE1QixJQUFJQyxJQUFKLENBQVNGLElBQVQsQ0FBYzhCLFNBQWQsQ0FBd0JQLFNBQVNJLE9BQWpDLEVBQTBDLGVBQTFDLENBQWI7QUFDQSxNQUZELE1BRU87QUFDTkMsYUFBT0csT0FBUCxDQUFlQyxJQUFmLENBQW9CLHFCQUFwQjtBQUNBO0FBQ0QsS0FURDtBQVVBLElBckJPO0FBc0JSQyxvQkF0QlEsK0JBc0JZO0FBQ25CLFFBQUloQixZQUFZLElBQWhCO0FBQ0EsV0FBT2hCLElBQUlpQixJQUFKLENBQVNDLEdBQVQsQ0FBYUMsR0FBYixDQUFpQjtBQUN2QkMsVUFBSztBQURrQixLQUFqQixFQUVKSyxJQUZJLENBRUMsb0JBQVk7QUFDbkJULGVBQVVoQyxRQUFWLEdBQXFCc0MsU0FBU1gsS0FBOUI7QUFDQSxLQUpNLENBQVA7QUFLQSxJQTdCTztBQThCUnNCLFVBOUJRLHFCQThCRTtBQUNULFFBQUksQ0FBQyxLQUFLbEQsVUFBVixFQUFzQjtBQUNyQixVQUFLRSxHQUFMLENBQVNFLGNBQVQsR0FBMEIsT0FBMUI7QUFDQSxVQUFLRixHQUFMLENBQVNNLGVBQVQsR0FBMkIsTUFBM0I7QUFDQSxVQUFLTixHQUFMLENBQVNJLFlBQVQsR0FBd0IsR0FBeEI7QUFDQTtBQUNELFFBQUkyQixZQUFZLElBQWhCO0FBQ0FoQixRQUFJaUIsSUFBSixDQUFTQyxHQUFULENBQWFNLElBQWIsQ0FBa0I7QUFDakJKLFVBQUsseUJBRFk7QUFFakJ4QyxXQUFNO0FBQ0xzRCxlQUFTbEIsVUFBVS9CO0FBRGQ7QUFGVyxLQUFsQixFQUtHd0MsSUFMSCxDQUtRLFVBQUNILFFBQUQsRUFBYztBQUNyQixTQUFJQSxTQUFTSSxPQUFiLEVBQXNCO0FBQ3JCQyxhQUFPQyxLQUFQLENBQWE1QixJQUFJQyxJQUFKLENBQVNGLElBQVQsQ0FBYzhCLFNBQWQsQ0FBd0JQLFNBQVNJLE9BQWpDLEVBQTBDLGVBQTFDLENBQWI7QUFDQSxNQUZELE1BRU87QUFDTlYsZ0JBQVUvQixHQUFWLENBQWNDLE9BQWQsR0FBd0IsRUFBeEI7QUFDQThCLGdCQUFVL0IsR0FBVixDQUFjRSxjQUFkLEdBQStCLFFBQS9CO0FBQ0E2QixnQkFBVS9CLEdBQVYsQ0FBY0csS0FBZCxHQUFzQixFQUF0QjtBQUNBNEIsZ0JBQVUvQixHQUFWLENBQWNJLFlBQWQsR0FBNkIsR0FBN0I7QUFDQTJCLGdCQUFVL0IsR0FBVixDQUFjSyxNQUFkLEdBQXVCLEVBQXZCO0FBQ0EwQixnQkFBVS9CLEdBQVYsQ0FBY00sZUFBZCxHQUFnQyxPQUFoQztBQUNBeUIsZ0JBQVVnQixpQkFBVixHQUNFUCxJQURGLENBQ08sWUFBTTtBQUNYVCxpQkFBVXJCLFVBQVYsQ0FBcUJDLElBQXJCLEdBQTRCb0IsVUFBVVosUUFBdEM7QUFDQSxPQUhGO0FBSUE7QUFDRCxLQXBCRDtBQXFCQSxJQTFETztBQTJEUitCLGFBM0RRLHNCQTJER0MsTUEzREgsRUEyRFc7QUFDbEIsUUFBSXBCLFlBQVksSUFBaEI7QUFDQWhCLFFBQUlpQixJQUFKLENBQVNDLEdBQVQsQ0FBYU0sSUFBYixDQUFrQjtBQUNqQkosVUFBSyw0QkFEWTtBQUVqQnhDLFdBQU07QUFDTHdELGNBQVFBO0FBREg7QUFGVyxLQUFsQixFQUtHWCxJQUxILENBS1EsVUFBQ0gsUUFBRCxFQUFjO0FBQ3JCLFNBQUlBLFNBQVNJLE9BQWIsRUFBc0I7QUFDckJDLGFBQU9DLEtBQVAsQ0FBYTVCLElBQUlDLElBQUosQ0FBU0YsSUFBVCxDQUFjOEIsU0FBZCxDQUF3QlAsU0FBU0ksT0FBakMsRUFBMEMsZUFBMUMsQ0FBYjtBQUNBO0FBQ0RWLGVBQVVnQixpQkFBVixHQUNFUCxJQURGLENBQ08sWUFBTTtBQUNYLFVBQUlULFVBQVVyQixVQUFWLENBQXFCQyxJQUFyQixHQUE0Qm9CLFVBQVVaLFFBQTFDLEVBQW9EO0FBQ25EWSxpQkFBVXJCLFVBQVYsQ0FBcUJDLElBQXJCLEdBQTRCb0IsVUFBVVosUUFBdEM7QUFDQTtBQUNELE1BTEY7QUFNQSxLQWZEO0FBZ0JBLElBN0VPO0FBOEVSaUMsZUE5RVEsd0JBOEVLQyxTQTlFTCxFQThFZ0JGLE1BOUVoQixFQThFd0I7QUFDL0IsUUFBSUUsY0FBYyxJQUFsQixFQUF3QjtBQUN2QixVQUFLQyxVQUFMLENBQWdCSCxNQUFoQjtBQUNBLEtBRkQsTUFFTztBQUNOLFVBQUtJLFdBQUwsQ0FBaUJKLE1BQWpCO0FBQ0E7QUFDRCxJQXBGTztBQXFGUkcsYUFyRlEsc0JBcUZHSCxNQXJGSCxFQXFGVztBQUNsQixRQUFJcEIsWUFBWSxJQUFoQjtBQUNBaEIsUUFBSWlCLElBQUosQ0FBU0MsR0FBVCxDQUFhTSxJQUFiLENBQWtCO0FBQ2pCSixVQUFLLDRCQURZO0FBRWpCeEMsV0FBTTtBQUNMd0QsY0FBUUE7QUFESDtBQUZXLEtBQWxCLEVBS0dYLElBTEgsQ0FLUSxVQUFDSCxRQUFELEVBQWM7QUFDckIsU0FBSUEsU0FBU0ksT0FBYixFQUFzQjtBQUNyQkMsYUFBT0MsS0FBUCxDQUFhNUIsSUFBSUMsSUFBSixDQUFTRixJQUFULENBQWM4QixTQUFkLENBQXdCUCxTQUFTSSxPQUFqQyxFQUEwQyxlQUExQyxDQUFiO0FBQ0E7QUFDRFYsZUFBVWdCLGlCQUFWO0FBQ0EsS0FWRDtBQVdBLElBbEdPO0FBbUdSUSxjQW5HUSx1QkFtR0lKLE1BbkdKLEVBbUdZO0FBQ25CLFFBQUlwQixZQUFZLElBQWhCO0FBQ0FoQixRQUFJaUIsSUFBSixDQUFTQyxHQUFULENBQWFNLElBQWIsQ0FBa0I7QUFDakJKLFVBQUssNkJBRFk7QUFFakJ4QyxXQUFNO0FBQ0x3RCxjQUFRQTtBQURIO0FBRlcsS0FBbEIsRUFLR1gsSUFMSCxDQUtRLFVBQUNILFFBQUQsRUFBYztBQUNyQixTQUFJQSxTQUFTSSxPQUFiLEVBQXNCO0FBQ3JCQyxhQUFPQyxLQUFQLENBQWE1QixJQUFJQyxJQUFKLENBQVNGLElBQVQsQ0FBYzhCLFNBQWQsQ0FBd0JQLFNBQVNJLE9BQWpDLEVBQTBDLGVBQTFDLENBQWI7QUFDQTtBQUNEVixlQUFVZ0IsaUJBQVY7QUFDQSxLQVZEO0FBV0EsSUFoSE87QUFpSFJTLFdBakhRLG9CQWlIQ0MsS0FqSEQsRUFpSFFDLFNBakhSLEVBaUhtQjtBQUMxQixRQUFJM0IsWUFBWSxJQUFoQjtBQUNBLFFBQUk0QixNQUFNRixNQUFNcEQsTUFBTixDQUFhdUQsT0FBYixDQUFxQixJQUFyQixDQUFWO0FBQ0EsUUFBSUMsUUFBUUosTUFBTXBELE1BQU4sQ0FBYXVELE9BQWIsQ0FBcUIsT0FBckIsQ0FBWjtBQUNBQyxVQUFNQyxnQkFBTixDQUF1QixTQUF2QixFQUFrQ0MsT0FBbEMsQ0FBMEMsVUFBU0MsRUFBVCxFQUFhO0FBQ3REQSxRQUFHQyxTQUFILENBQWFDLE1BQWIsQ0FBb0IsTUFBcEI7QUFDQSxLQUZEO0FBR0EsU0FBSzNELElBQUwsR0FBWSxLQUFLbUIsS0FBTCxDQUFXZ0MsU0FBWCxDQUFaO0FBQ0EsUUFBSSxLQUFLbkQsSUFBTCxDQUFVSixLQUFkLEVBQXFCO0FBQ3BCLFVBQUtJLElBQUwsQ0FBVU4sT0FBVixJQUFxQixNQUFNLEtBQUtNLElBQUwsQ0FBVUosS0FBckM7QUFDQSxVQUFLSSxJQUFMLENBQVVKLEtBQVYsR0FBa0IsRUFBbEI7QUFDQTtBQUNEd0QsUUFBSU0sU0FBSixDQUFjakUsR0FBZCxDQUFrQixNQUFsQjtBQUNBLElBOUhPO0FBK0hSbUUsV0EvSFEsb0JBK0hDVixLQS9IRCxFQStIUTtBQUNmLFFBQUkxQixZQUFZLElBQWhCO0FBQ0FoQixRQUFJaUIsSUFBSixDQUFTQyxHQUFULENBQWFNLElBQWIsQ0FBa0I7QUFDakJKLFVBQUssNEJBRFk7QUFFakJ4QyxXQUFNO0FBQ0x5RSxnQkFBVXJDLFVBQVV4QjtBQURmO0FBRlcsS0FBbEIsRUFLR2lDLElBTEgsQ0FLUSxVQUFDSCxRQUFELEVBQWM7QUFDckIsU0FBSUEsU0FBU0ksT0FBYixFQUFzQjtBQUNyQkMsYUFBT0MsS0FBUCxDQUFhNUIsSUFBSUMsSUFBSixDQUFTRixJQUFULENBQWM4QixTQUFkLENBQXdCUCxTQUFTSSxPQUFqQyxFQUEwQyxlQUExQyxDQUFiO0FBQ0EsTUFGRCxNQUVPO0FBQ04sVUFBSW9CLFFBQVFKLE1BQU1wRCxNQUFOLENBQWF1RCxPQUFiLENBQXFCLE9BQXJCLENBQVo7QUFDQUMsWUFBTUMsZ0JBQU4sQ0FBdUIsU0FBdkIsRUFBa0NDLE9BQWxDLENBQTBDLFVBQVNDLEVBQVQsRUFBYTtBQUN0REEsVUFBR0MsU0FBSCxDQUFhQyxNQUFiLENBQW9CLE1BQXBCO0FBQ0EsT0FGRDtBQUdBbkMsZ0JBQVVnQixpQkFBVjtBQUNBO0FBQ0QsS0FmRDtBQWdCQTtBQWpKTyxHQXhESDtBQTJNTnNCLFNBM01NLHFCQTJNSTtBQUNULFFBQUt2QyxpQkFBTDtBQUNBLFFBQUtpQixpQkFBTDtBQUNBLEdBOU1LOztBQStNTnVCLFFBQU07QUFBQSxVQUFRbEMsTUFBUjtBQUFBO0FBL01BLEVBQVA7QUFpTkEsQ0FyTkYiLCJmaWxlIjoicmVkaXJlY3RydWxlcy9yZWRpcmVjdHJ1bGVzLW92ZXJ2aWV3LmpzIiwic291cmNlc0NvbnRlbnQiOlsicmVkaXJlY3RydWxlcy5jb250cm9sbGVycy5tb2R1bGUoXG5cdCdyZWRpcmVjdHJ1bGVzLW92ZXJ2aWV3Jyxcblx0Wyd4aHInXSxcblx0KCkgPT4ge1xuXHRcdHJldHVybiB7XG5cdFx0XHR2dWU6IHRydWUsXG5cdFx0XHRuYW1lOiAnZ3gtcmVkaXJlY3RydWxlcy1vdmVydmlldycsXG5cdFx0XHR0ZW1wbGF0ZTogJyNyZWRpcmVjdHJ1bGVzLW92ZXJ2aWV3Jyxcblx0XHRcdHByb3BzOiBbJ2Jhc2VfdXJsJ10sXG5cdFx0XHRkYXRhOiAoKSA9PiAoe1xuXHRcdFx0XHRjb25maWd1cmF0aW9uOiB7XG5cdFx0XHRcdFx0bWF4QWdlOiAzNjAwLFxuXHRcdFx0XHR9LFxuXHRcdFx0XHRleHBlcnRNb2RlOiBmYWxzZSxcblx0XHRcdFx0YWxsUnVsZXM6IFtdLFxuXHRcdFx0XHQvL3J1bGVzOiBbXSxcblx0XHRcdFx0YWRkOiB7XG5cdFx0XHRcdFx0dXJsUGF0aDogJycsXG5cdFx0XHRcdFx0cXVlcnlNYXRjaE1vZGU6ICdtYXRjaCcsXG5cdFx0XHRcdFx0cXVlcnk6ICcnLFxuXHRcdFx0XHRcdHJlc3BvbnNlQ29kZTogMzAxLFxuXHRcdFx0XHRcdHRhcmdldDogJycsXG5cdFx0XHRcdFx0cXVlcnlQcm9jZXNzaW5nOiAnZHJvcCcsXG5cdFx0XHRcdH0sXG5cdFx0XHRcdGVkaXQ6IHtcblx0XHRcdFx0XHRpZDogMCxcblx0XHRcdFx0XHR1cmxQYXRoOiAnJyxcblx0XHRcdFx0XHRxdWVyeU1hdGNoTW9kZTogJ21hdGNoJyxcblx0XHRcdFx0XHRxdWVyeTogJycsXG5cdFx0XHRcdFx0cmVzcG9uc2VDb2RlOiAzMDEsXG5cdFx0XHRcdFx0dGFyZ2V0OiAnJyxcblx0XHRcdFx0XHRxdWVyeVByb2Nlc3Npbmc6ICdkcm9wJyxcblx0XHRcdFx0XHRzdGF0dXM6IHRydWUsXG5cdFx0XHRcdFx0bWF4QWdlOiAzNjAwXG5cdFx0XHRcdH0sXG5cdFx0XHRcdHBhZ2luYXRpb246IHtcblx0XHRcdFx0XHRwYWdlOiAxLFxuXHRcdFx0XHRcdGVsZW1lbnRzUGVyUGFnZTogMTAsXG5cdFx0XHRcdFx0bGltaXRzOiBbMTAsIDI1LCA1MCwgMTAwLCAyMDBdXG5cdFx0XHRcdH0sXG5cdFx0XHRcdGxhbmc6IGpzZS5jb3JlLmxhbmcuZ2V0U2VjdGlvbigncmVkaXJlY3RydWxlcycpLFxuXHRcdFx0fSksXG5cdFx0XHRjb21wdXRlZDoge1xuXHRcdFx0XHRtYXhQYWdlcygpIHtcblx0XHRcdFx0XHRyZXR1cm4gTWF0aC5jZWlsKHRoaXMuYWxsUnVsZXMubGVuZ3RoIC8gdGhpcy5wYWdpbmF0aW9uLmVsZW1lbnRzUGVyUGFnZSk7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdGN1cnJlbnRQYWdlU3RhcnQoKSB7XG5cdFx0XHRcdFx0cmV0dXJuICh0aGlzLnBhZ2luYXRpb24ucGFnZSAtIDEpICogdGhpcy5wYWdpbmF0aW9uLmVsZW1lbnRzUGVyUGFnZTtcblx0XHRcdFx0fSxcblx0XHRcdFx0Y3VycmVudFBhZ2VFbmQoKSB7XG5cdFx0XHRcdFx0cmV0dXJuIHRoaXMuY3VycmVudFBhZ2VTdGFydCArIHRoaXMucGFnaW5hdGlvbi5lbGVtZW50c1BlclBhZ2U7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdHRvdGFsRWxlbWVudHNDb3VudCgpIHtcblx0XHRcdFx0XHRyZXR1cm4gdGhpcy5hbGxSdWxlcy5sZW5ndGg7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdHJ1bGVzKCkge1xuXHRcdFx0XHRcdHJldHVybiB0aGlzLmFsbFJ1bGVzLnNsaWNlKHRoaXMuY3VycmVudFBhZ2VTdGFydCwgdGhpcy5jdXJyZW50UGFnZUVuZCk7XG5cdFx0XHRcdH1cblx0XHRcdH0sXG5cdFx0XHR3YXRjaDoge30sXG5cdFx0XHRtZXRob2RzOiB7XG5cdFx0XHRcdGxvYWRDb25maWd1cmF0aW9uKCkge1xuXHRcdFx0XHRcdGxldCBjb21wb25lbnQgPSB0aGlzO1xuXHRcdFx0XHRcdGpzZS5saWJzLnhoci5nZXQoe1xuXHRcdFx0XHRcdFx0dXJsOiAncmVkaXJlY3QtcnVsZXMvZ2V0LWNvbmZpZ3VyYXRpb24nXG5cdFx0XHRcdFx0fSkuZG9uZShyZXNwb25zZSA9PiB7XG5cdFx0XHRcdFx0XHRjb21wb25lbnQuY29uZmlndXJhdGlvbiA9IHJlc3BvbnNlO1xuXHRcdFx0XHRcdH0pO1xuXHRcdFx0XHR9LFxuXHRcdFx0XHRzYXZlQ29uZmlndXJhdGlvbigpIHtcblx0XHRcdFx0XHRsZXQgY29tcG9uZW50ID0gdGhpcztcblx0XHRcdFx0XHRqc2UubGlicy54aHIucG9zdCh7XG5cdFx0XHRcdFx0XHR1cmw6ICdyZWRpcmVjdC1ydWxlcy9zYXZlLWNvbmZpZ3VyYXRpb24nLFxuXHRcdFx0XHRcdFx0ZGF0YTogY29tcG9uZW50LmNvbmZpZ3VyYXRpb25cblx0XHRcdFx0XHR9KS50aGVuKChyZXNwb25zZSkgPT4ge1xuXHRcdFx0XHRcdFx0aWYgKHJlc3BvbnNlLm1lc3NhZ2UpIHtcblx0XHRcdFx0XHRcdFx0d2luZG93LmFsZXJ0KGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKHJlc3BvbnNlLm1lc3NhZ2UsICdyZWRpcmVjdHJ1bGVzJykpO1xuXHRcdFx0XHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0XHRcdFx0d2luZG93LmNvbnNvbGUuaW5mbygnY29uZmlndXJhdGlvbiBzYXZlZCcpO1xuXHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdH0pO1xuXHRcdFx0XHR9LFxuXHRcdFx0XHRsb2FkUmVkaXJlY3RSdWxlcygpIHtcblx0XHRcdFx0XHRsZXQgY29tcG9uZW50ID0gdGhpcztcblx0XHRcdFx0XHRyZXR1cm4ganNlLmxpYnMueGhyLmdldCh7XG5cdFx0XHRcdFx0XHR1cmw6ICdyZWRpcmVjdC1ydWxlcy9nZXQtcnVsZXMnXG5cdFx0XHRcdFx0fSkudGhlbihyZXNwb25zZSA9PiB7XG5cdFx0XHRcdFx0XHRjb21wb25lbnQuYWxsUnVsZXMgPSByZXNwb25zZS5ydWxlcztcblx0XHRcdFx0XHR9KTtcblx0XHRcdFx0fSxcblx0XHRcdFx0YWRkUnVsZSgpIHtcblx0XHRcdFx0XHRpZiAoIXRoaXMuZXhwZXJ0TW9kZSkge1xuXHRcdFx0XHRcdFx0dGhpcy5hZGQucXVlcnlNYXRjaE1vZGUgPSAnbWF0Y2gnO1xuXHRcdFx0XHRcdFx0dGhpcy5hZGQucXVlcnlQcm9jZXNzaW5nID0gJ2Ryb3AnO1xuXHRcdFx0XHRcdFx0dGhpcy5hZGQucmVzcG9uc2VDb2RlID0gMzAxO1xuXHRcdFx0XHRcdH1cblx0XHRcdFx0XHRsZXQgY29tcG9uZW50ID0gdGhpcztcblx0XHRcdFx0XHRqc2UubGlicy54aHIucG9zdCh7XG5cdFx0XHRcdFx0XHR1cmw6ICdyZWRpcmVjdC1ydWxlcy9hZGQtcnVsZScsXG5cdFx0XHRcdFx0XHRkYXRhOiB7XG5cdFx0XHRcdFx0XHRcdG5ld1J1bGU6IGNvbXBvbmVudC5hZGRcblx0XHRcdFx0XHRcdH1cblx0XHRcdFx0XHR9KS50aGVuKChyZXNwb25zZSkgPT4ge1xuXHRcdFx0XHRcdFx0aWYgKHJlc3BvbnNlLm1lc3NhZ2UpIHtcblx0XHRcdFx0XHRcdFx0d2luZG93LmFsZXJ0KGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKHJlc3BvbnNlLm1lc3NhZ2UsICdyZWRpcmVjdHJ1bGVzJykpO1xuXHRcdFx0XHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0XHRcdFx0Y29tcG9uZW50LmFkZC51cmxQYXRoID0gJyc7XG5cdFx0XHRcdFx0XHRcdGNvbXBvbmVudC5hZGQucXVlcnlNYXRjaE1vZGUgPSAnaWdub3JlJztcblx0XHRcdFx0XHRcdFx0Y29tcG9uZW50LmFkZC5xdWVyeSA9ICcnO1xuXHRcdFx0XHRcdFx0XHRjb21wb25lbnQuYWRkLnJlc3BvbnNlQ29kZSA9IDMwMjtcblx0XHRcdFx0XHRcdFx0Y29tcG9uZW50LmFkZC50YXJnZXQgPSAnJztcblx0XHRcdFx0XHRcdFx0Y29tcG9uZW50LmFkZC5xdWVyeVByb2Nlc3NpbmcgPSAnbWVyZ2UnO1xuXHRcdFx0XHRcdFx0XHRjb21wb25lbnQubG9hZFJlZGlyZWN0UnVsZXMoKVxuXHRcdFx0XHRcdFx0XHRcdC50aGVuKCgpID0+IHtcblx0XHRcdFx0XHRcdFx0XHRcdGNvbXBvbmVudC5wYWdpbmF0aW9uLnBhZ2UgPSBjb21wb25lbnQubWF4UGFnZXM7XG5cdFx0XHRcdFx0XHRcdFx0fSk7XG5cdFx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0fSk7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdGRlbGV0ZVJ1bGUocnVsZUlkKSB7XG5cdFx0XHRcdFx0bGV0IGNvbXBvbmVudCA9IHRoaXM7XG5cdFx0XHRcdFx0anNlLmxpYnMueGhyLnBvc3Qoe1xuXHRcdFx0XHRcdFx0dXJsOiAncmVkaXJlY3QtcnVsZXMvZGVsZXRlLXJ1bGUnLFxuXHRcdFx0XHRcdFx0ZGF0YToge1xuXHRcdFx0XHRcdFx0XHRydWxlSWQ6IHJ1bGVJZFxuXHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdH0pLnRoZW4oKHJlc3BvbnNlKSA9PiB7XG5cdFx0XHRcdFx0XHRpZiAocmVzcG9uc2UubWVzc2FnZSkge1xuXHRcdFx0XHRcdFx0XHR3aW5kb3cuYWxlcnQoanNlLmNvcmUubGFuZy50cmFuc2xhdGUocmVzcG9uc2UubWVzc2FnZSwgJ3JlZGlyZWN0cnVsZXMnKSk7XG5cdFx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0XHRjb21wb25lbnQubG9hZFJlZGlyZWN0UnVsZXMoKVxuXHRcdFx0XHRcdFx0XHQudGhlbigoKSA9PiB7XG5cdFx0XHRcdFx0XHRcdFx0aWYgKGNvbXBvbmVudC5wYWdpbmF0aW9uLnBhZ2UgPiBjb21wb25lbnQubWF4UGFnZXMpIHtcblx0XHRcdFx0XHRcdFx0XHRcdGNvbXBvbmVudC5wYWdpbmF0aW9uLnBhZ2UgPSBjb21wb25lbnQubWF4UGFnZXM7XG5cdFx0XHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdFx0XHR9KTtcblx0XHRcdFx0XHR9KTtcblx0XHRcdFx0fSxcblx0XHRcdFx0Y2hhbmdlU3RhdHVzKG5ld1N0YXR1cywgcnVsZUlkKSB7XG5cdFx0XHRcdFx0aWYgKG5ld1N0YXR1cyA9PT0gdHJ1ZSkge1xuXHRcdFx0XHRcdFx0dGhpcy5lbmFibGVSdWxlKHJ1bGVJZCk7XG5cdFx0XHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0XHRcdHRoaXMuZGlzYWJsZVJ1bGUocnVsZUlkKTtcblx0XHRcdFx0XHR9XG5cdFx0XHRcdH0sXG5cdFx0XHRcdGVuYWJsZVJ1bGUocnVsZUlkKSB7XG5cdFx0XHRcdFx0bGV0IGNvbXBvbmVudCA9IHRoaXM7XG5cdFx0XHRcdFx0anNlLmxpYnMueGhyLnBvc3Qoe1xuXHRcdFx0XHRcdFx0dXJsOiAncmVkaXJlY3QtcnVsZXMvZW5hYmxlLXJ1bGUnLFxuXHRcdFx0XHRcdFx0ZGF0YToge1xuXHRcdFx0XHRcdFx0XHRydWxlSWQ6IHJ1bGVJZFxuXHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdH0pLnRoZW4oKHJlc3BvbnNlKSA9PiB7XG5cdFx0XHRcdFx0XHRpZiAocmVzcG9uc2UubWVzc2FnZSkge1xuXHRcdFx0XHRcdFx0XHR3aW5kb3cuYWxlcnQoanNlLmNvcmUubGFuZy50cmFuc2xhdGUocmVzcG9uc2UubWVzc2FnZSwgJ3JlZGlyZWN0cnVsZXMnKSk7XG5cdFx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0XHRjb21wb25lbnQubG9hZFJlZGlyZWN0UnVsZXMoKTtcblx0XHRcdFx0XHR9KTtcblx0XHRcdFx0fSxcblx0XHRcdFx0ZGlzYWJsZVJ1bGUocnVsZUlkKSB7XG5cdFx0XHRcdFx0bGV0IGNvbXBvbmVudCA9IHRoaXM7XG5cdFx0XHRcdFx0anNlLmxpYnMueGhyLnBvc3Qoe1xuXHRcdFx0XHRcdFx0dXJsOiAncmVkaXJlY3QtcnVsZXMvZGlzYWJsZS1ydWxlJyxcblx0XHRcdFx0XHRcdGRhdGE6IHtcblx0XHRcdFx0XHRcdFx0cnVsZUlkOiBydWxlSWRcblx0XHRcdFx0XHRcdH1cblx0XHRcdFx0XHR9KS50aGVuKChyZXNwb25zZSkgPT4ge1xuXHRcdFx0XHRcdFx0aWYgKHJlc3BvbnNlLm1lc3NhZ2UpIHtcblx0XHRcdFx0XHRcdFx0d2luZG93LmFsZXJ0KGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKHJlc3BvbnNlLm1lc3NhZ2UsICdyZWRpcmVjdHJ1bGVzJykpO1xuXHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdFx0Y29tcG9uZW50LmxvYWRSZWRpcmVjdFJ1bGVzKCk7XG5cdFx0XHRcdFx0fSk7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdGVkaXRSdWxlKGV2ZW50LCBydWxlaW5kZXgpIHtcblx0XHRcdFx0XHRsZXQgY29tcG9uZW50ID0gdGhpcztcblx0XHRcdFx0XHRsZXQgcm93ID0gZXZlbnQudGFyZ2V0LmNsb3Nlc3QoJ3RyJyk7XG5cdFx0XHRcdFx0bGV0IHRhYmxlID0gZXZlbnQudGFyZ2V0LmNsb3Nlc3QoJ3RhYmxlJyk7XG5cdFx0XHRcdFx0dGFibGUucXVlcnlTZWxlY3RvckFsbCgndHIuZWRpdCcpLmZvckVhY2goZnVuY3Rpb24oZWwpIHtcblx0XHRcdFx0XHRcdGVsLmNsYXNzTGlzdC5yZW1vdmUoJ2VkaXQnKTtcblx0XHRcdFx0XHR9KTtcblx0XHRcdFx0XHR0aGlzLmVkaXQgPSB0aGlzLnJ1bGVzW3J1bGVpbmRleF07XG5cdFx0XHRcdFx0aWYgKHRoaXMuZWRpdC5xdWVyeSkge1xuXHRcdFx0XHRcdFx0dGhpcy5lZGl0LnVybFBhdGggKz0gJz8nICsgdGhpcy5lZGl0LnF1ZXJ5O1xuXHRcdFx0XHRcdFx0dGhpcy5lZGl0LnF1ZXJ5ID0gJyc7XG5cdFx0XHRcdFx0fVxuXHRcdFx0XHRcdHJvdy5jbGFzc0xpc3QuYWRkKCdlZGl0Jyk7XG5cdFx0XHRcdH0sXG5cdFx0XHRcdHNhdmVSdWxlKGV2ZW50KSB7XG5cdFx0XHRcdFx0bGV0IGNvbXBvbmVudCA9IHRoaXM7XG5cdFx0XHRcdFx0anNlLmxpYnMueGhyLnBvc3Qoe1xuXHRcdFx0XHRcdFx0dXJsOiAncmVkaXJlY3QtcnVsZXMvdXBkYXRlLXJ1bGUnLFxuXHRcdFx0XHRcdFx0ZGF0YToge1xuXHRcdFx0XHRcdFx0XHRydWxlRGF0YTogY29tcG9uZW50LmVkaXRcblx0XHRcdFx0XHRcdH1cblx0XHRcdFx0XHR9KS50aGVuKChyZXNwb25zZSkgPT4ge1xuXHRcdFx0XHRcdFx0aWYgKHJlc3BvbnNlLm1lc3NhZ2UpIHtcblx0XHRcdFx0XHRcdFx0d2luZG93LmFsZXJ0KGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKHJlc3BvbnNlLm1lc3NhZ2UsICdyZWRpcmVjdHJ1bGVzJykpO1xuXHRcdFx0XHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0XHRcdFx0bGV0IHRhYmxlID0gZXZlbnQudGFyZ2V0LmNsb3Nlc3QoJ3RhYmxlJyk7XG5cdFx0XHRcdFx0XHRcdHRhYmxlLnF1ZXJ5U2VsZWN0b3JBbGwoJ3RyLmVkaXQnKS5mb3JFYWNoKGZ1bmN0aW9uKGVsKSB7XG5cdFx0XHRcdFx0XHRcdFx0ZWwuY2xhc3NMaXN0LnJlbW92ZSgnZWRpdCcpO1xuXHRcdFx0XHRcdFx0XHR9KTtcblx0XHRcdFx0XHRcdFx0Y29tcG9uZW50LmxvYWRSZWRpcmVjdFJ1bGVzKCk7XG5cdFx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0fSk7XG5cdFx0XHRcdH1cblx0XHRcdH0sXG5cdFx0XHRtb3VudGVkKCkge1xuXHRcdFx0XHR0aGlzLmxvYWRDb25maWd1cmF0aW9uKCk7XG5cdFx0XHRcdHRoaXMubG9hZFJlZGlyZWN0UnVsZXMoKTtcblx0XHRcdH0sXG5cdFx0XHRpbml0OiBkb25lID0+IGRvbmUoKVxuXHRcdH07XG5cdH1cbik7XG5cblxuIl19
