'use strict';

/* --------------------------------------------------------------
 orders_table_controller.js 2023-07-07
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2023 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Orders Table Controller
 *
 * This controller contains the mapping logic of the orders table.
 *
 * @module Compatibility/orders_table_controller
 */
gx.compatibility.module('orders_table_controller', [gx.source + '/libs/action_mapper', gx.source + '/libs/button_dropdown'],

/**  @lends module:Compatibility/orders_table_controller */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Array of mapped buttons
     *
     * @var Array
     */
    mappedButtons = [],


    /**
     * The mapper library
     *
     * @var {object}
     */
    mapper = jse.libs.action_mapper,


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------

    /**
     * Disable/Enable the buttons on the bottom button-dropdown
     * dependent on the checkboxes selection
     * @private
     */
    var _toggleMultiActionButton = function _toggleMultiActionButton() {
        var $checked = $('tr[data-row-id] input[type="checkbox"]:checked');
        $('.js-bottom-dropdown button').prop('disabled', !$checked.length);
    };

    /**
     * Map actions for every row in the table.
     *
     * This method will map the actions for each
     * row of the table.
     *
     * @private
     */
    var _mapRowAction = function _mapRowAction($that) {
        /**
         * Reference to the row action dropdown
         * @var {object | jQuery}
         */
        var $dropdown = $that.find('.js-button-dropdown');

        if ($dropdown.length) {
            _mapRowButtonDropdown($dropdown);
        }
    };

    var _mapRowButtonDropdown = function _mapRowButtonDropdown($dropdown) {
        var actions = ['TEXT_SHOW', 'TEXT_GM_STATUS', 'delete', 'BUTTON_GM_CANCEL', 'BUTTON_CREATE_INVOICE', 'TITLE_INVOICE_MAIL', 'BUTTON_CREATE_PACKING_SLIP', 'TITLE_SEND_ORDER', 'TEXT_CREATE_WITHDRAWAL', 'TXT_PARCEL_TRACKING_SENDBUTTON_TITLE', 'BUTTON_DHL_LABEL', 'MAILBEEZ_OVERVIEW', 'MAILBEEZ_NOTIFICATIONS', 'MAILBEEZ_CONVERSATIONS', 'BUTTON_HERMES'];

        for (var index in actions) {
            _bindEventHandler($dropdown, actions[index], '.single-order-dropdown');
        }
    };

    /**
     * Defines the language section for each text tile
     *
     * @type {object}
     * @private
     */
    var _sectionMapping = {
        'TEXT_SHOW': 'orders',
        'TEXT_GM_STATUS': 'orders',
        'delete': 'buttons',
        'BUTTON_GM_CANCEL': 'orders',
        'BUTTON_CREATE_INVOICE': 'orders',
        'TITLE_INVOICE_MAIL': 'orders',
        'BUTTON_CREATE_PACKING_SLIP': 'orders',
        'TITLE_SEND_ORDER': 'orders',
        'TEXT_CREATE_WITHDRAWAL': 'orders',
        'TXT_PARCEL_TRACKING_SENDBUTTON_TITLE': 'parcel_services',
        'BUTTON_DHL_LABEL': 'orders',
        'MAILBEEZ_OVERVIEW': 'orders',
        'MAILBEEZ_NOTIFICATIONS': 'orders',
        'MAILBEEZ_CONVERSATIONS': 'orders',
        'BUTTON_MULTI_CANCEL': 'orders',
        'BUTTON_MULTI_CHANGE_ORDER_STATUS': 'orders',
        'BUTTON_MULTI_DELETE': 'orders',
        'BUTTON_HERMES': 'orders',
        'get_labels': 'iloxx'
    };

    /**
     * Defines target selectors
     *
     * @type {object}
     * @private
     */
    var _selectorMapping = {
        'TEXT_SHOW': '.contentTable .infoBoxContent a.btn-details',
        'TEXT_GM_STATUS': '.contentTable .infoBoxContent a.btn-update_order_status',
        'delete': '.contentTable .infoBoxContent a.btn-delete',
        'BUTTON_GM_CANCEL': '.contentTable .infoBoxContent .GM_CANCEL',
        'BUTTON_CREATE_INVOICE': '.contentTable .infoBoxContent a.btn-invoice',
        'TITLE_INVOICE_MAIL': '.contentTable .infoBoxContent .GM_INVOICE_MAIL',
        'BUTTON_CREATE_PACKING_SLIP': '.contentTable .infoBoxContent a.btn-packing_slip',
        'TITLE_SEND_ORDER': '.contentTable .infoBoxContent .GM_SEND_ORDER',
        'TEXT_CREATE_WITHDRAWAL': '.contentTable .infoBoxContent a.btn-create_withdrawal',
        'TXT_PARCEL_TRACKING_SENDBUTTON_TITLE': '.contentTable .infoBoxContent a.btn-add_tracking_code',
        'BUTTON_DHL_LABEL': '.contentTable .infoBoxContent a.btn-dhl_label',
        'MAILBEEZ_OVERVIEW': '.contentTable .infoBoxContent a.context_view_button.btn_left',
        'MAILBEEZ_NOTIFICATIONS': '.contentTable .infoBoxContent a.context_view_button.btn_middle',
        'MAILBEEZ_CONVERSATIONS': '.contentTable .infoBoxContent a.context_view_button.btn_right',
        'BUTTON_MULTI_CANCEL': '.contentTable .infoBoxContent a.btn-multi_cancel',
        'BUTTON_MULTI_CHANGE_ORDER_STATUS': '.contentTable .infoBoxContent a.btn-update_order_status',
        'BUTTON_MULTI_DELETE': '.contentTable .infoBoxContent a.btn-multi_delete',
        'BUTTON_HERMES': '.contentTable .infoBoxContent a.btn-hermes',
        'get_labels': '#iloxx_orders'
    };

    var _getActionCallback = function _getActionCallback(action) {
        switch (action) {
            case 'TEXT_SHOW':
                return _showOrderCallback;
            case 'TEXT_GM_STATUS':
                return _changeOrderStatusCallback;
            case 'delete':
                return _deleteCallback;
            case 'BUTTON_GM_CANCEL':
                return _cancelCallback;
            case 'BUTTON_CREATE_INVOICE':
                return _invoiceCallback;
            case 'TITLE_INVOICE_MAIL':
                return _emailInvoiceCallback;
            case 'BUTTON_CREATE_PACKING_SLIP':
                return _packingSlipCallback;
            case 'TITLE_SEND_ORDER':
                return _sendOrderConfirmationCallback;
            case 'TEXT_CREATE_WITHDRAWAL':
                return _withdrawalCallback;
            case 'TXT_PARCEL_TRACKING_SENDBUTTON_TITLE':
                return _addTrackingCodeCallback;
            case 'BUTTON_DHL_LABEL':
                return _dhlLabelCallback;
            case 'MAILBEEZ_OVERVIEW':
                return _mailBeezOverviewCallback;
            case 'MAILBEEZ_NOTIFICATIONS':
                return _mailBeezNotificationsCallback;
            case 'MAILBEEZ_CONVERSATIONS':
                return _mailBeezConversationsCallback;
            case 'BUTTON_MULTI_CANCEL':
                return _multiCancelCallback;
            case 'BUTTON_MULTI_CHANGE_ORDER_STATUS':
                return _multiChangeOrderStatusCallback;
            case 'BUTTON_MULTI_DELETE':
                return _multiDeleteCallback;
            case 'BUTTON_HERMES':
                return _hermesCallback;
            case 'get_labels':
                return _iloxxCallback;
        }
    };

    var _showOrderCallback = function _showOrderCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var url = $(_selectorMapping.TEXT_SHOW).attr('href');
        window.open(url.replace(/oID=(.*)&/, 'oID=' + orderId + '&'), '_self');
    };

    var _changeOrderStatusCallback = function _changeOrderStatusCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        $('#gm_order_id').val(orderId);
        $('.gx-orders-table .single-checkbox').removeClass('checked');
        $('.gx-orders-table input:checkbox').prop('checked', false);
        $(event.target).parents('tr').eq(0).find('.single-checkbox').addClass('checked');
        $(event.target).parents('tr').eq(0).find('input:checkbox').prop('checked', true);
        $(_selectorMapping.TEXT_GM_STATUS).click();
    };

    var _deleteCallback = function _deleteCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $delete = $(_selectorMapping.delete);
        $delete.data('order_id', orderId);
        $delete.get(0).click();
    };

    var _cancelCallback = function _cancelCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        $('#gm_order_id').val(orderId);
        $('.gx-orders-table .single-checkbox').removeClass('checked');
        $('.gx-orders-table input:checkbox').prop('checked', false);
        $(event.target).parents('tr').eq(0).find('.single-checkbox').addClass('checked');
        $(event.target).parents('tr').eq(0).find('input:checkbox').prop('checked', true);
        $(_selectorMapping.BUTTON_MULTI_CANCEL).click();
    };

    var _invoiceCallback = function _invoiceCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var url = $(_selectorMapping.BUTTON_CREATE_INVOICE).attr('href');
        window.open(url.replace(/oID=(.*)&/, 'oID=' + orderId + '&'), '_blank');
    };

    var _emailInvoiceCallback = function _emailInvoiceCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        $('#gm_order_id').val(orderId);
        $('.GM_INVOICE_MAIL').click();
    };

    var _packingSlipCallback = function _packingSlipCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var url = $(_selectorMapping.BUTTON_CREATE_PACKING_SLIP).attr('href');
        window.open(url.replace(/oID=(.*)&/, 'oID=' + orderId + '&'), '_blank');
    };

    var _sendOrderConfirmationCallback = function _sendOrderConfirmationCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        $('#gm_order_id').val(orderId);
        $('.GM_SEND_ORDER').click();
    };

    var _withdrawalCallback = function _withdrawalCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var url = $(_selectorMapping.TEXT_CREATE_WITHDRAWAL).attr('href');
        window.open(url.replace(/order=[^&]*/, 'order_id=' + orderId), '_blank');
    };

    var _addTrackingCodeCallback = function _addTrackingCodeCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $target = $(_selectorMapping.TXT_PARCEL_TRACKING_SENDBUTTON_TITLE);
        $target.data('order_id', orderId);
        $target.get(0).click();
    };

    var _dhlLabelCallback = function _dhlLabelCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var url = $(_selectorMapping.BUTTON_DHL_LABEL).attr('href');
        window.open(url.replace(/oID=(.*)/, 'oID=' + orderId), '_blank');
    };

    var _mailBeezOverviewCallback = function _mailBeezOverviewCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $target = $(_selectorMapping.MAILBEEZ_OVERVIEW);
        var url = $target.attr('onclick');
        url = url.replace(/oID=(.*)&/, 'oID=' + orderId + '&');
        $target.attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezNotificationsCallback = function _mailBeezNotificationsCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $target = $(_selectorMapping.MAILBEEZ_NOTIFICATIONS);
        var url = $target.attr('onclick');
        url = url.replace(/oID=(.*)&/, 'oID=' + orderId + '&');
        $target.attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezConversationsCallback = function _mailBeezConversationsCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $target = $(_selectorMapping.MAILBEEZ_CONVERSATIONS);
        var url = $target.attr('onclick');
        url = url.replace(/oID=(.*)&/, 'oID=' + orderId + '&');
        $target.attr('onclick', url);
        $target.get(0).click();
    };

    var _hermesCallback = function _hermesCallback(event) {
        var orderId = $(event.target).parents('tr').data('row-id');
        var $target = $(_selectorMapping.BUTTON_HERMES);
        var url = $target.attr('href');
        url = url.replace(/orders_id=(.*)/, 'orders_id=' + orderId);
        $target.attr('href', url);
        $target.get(0).click();
    };

    var _iloxxCallback = function _iloxxCallback(event) {
        var $target = $(_selectorMapping.get_labels);
        $target.click();
    };

    var _multiChangeOrderStatusCallback = function _multiChangeOrderStatusCallback(event) {
        $(_selectorMapping.BUTTON_MULTI_CHANGE_ORDER_STATUS).get(0).click();
    };

    var _multiDeleteCallback = function _multiDeleteCallback(event) {
        $(_selectorMapping.BUTTON_MULTI_DELETE).get(0).click();
    };

    var _multiCancelCallback = function _multiCancelCallback(event) {
        $(_selectorMapping.BUTTON_MULTI_CANCEL).get(0).click();
    };

    /**
     * Map table actions to bottom dropdown button.
     *
     * @private
     */
    var _mapTableActions = function _mapTableActions() {
        var $dropdown = $('#orders-table-dropdown');

        _bindEventHandler($dropdown, 'BUTTON_MULTI_CHANGE_ORDER_STATUS');

        if ($(_selectorMapping.get_labels).length) {
            _bindEventHandler($dropdown, 'get_labels');
        }

        _bindEventHandler($dropdown, 'BUTTON_MULTI_DELETE');
        _bindEventHandler($dropdown, 'BUTTON_MULTI_CANCEL');
    };

    /**
     * Map actions for every row in the table generically.
     *
     * This method will use the action_mapper library to map the actions for each
     * row of the table. It maps only those buttons, that haven't already explicitly
     * mapped by the _mapRowActions function.
     *
     * @private
     */
    var _mapUnmappedRowActions = function _mapUnmappedRowActions($this) {
        var unmappedRowActions = [];
        $('.action_buttons .extended_single_actions a,' + '.action_buttons .extended_single_actions button,' + '.action_buttons .extended_single_actions input[type="button"],' + '.action_buttons .extended_single_actions input[type="submit"]').each(function () {
            if (!_alreadyMapped($(this))) {
                unmappedRowActions.push($(this));
            }
        });

        var orderId = $this.data('row-id'),
            $dropdown = $this.find('.js-button-dropdown');

        $.each(unmappedRowActions, function () {
            var $button = $(this);
            var callback = function callback() {
                if ($button.prop('href') !== undefined) {
                    $button.prop('href', $button.prop('href').replace(/oID=(.*)\d(?=&)?/, 'oID=' + orderId));
                }
                $button.get(0).click();
            };

            jse.libs.button_dropdown.mapAction($dropdown, $button.text(), '', callback);
            mappedButtons.push($button);
        });
    };

    var _mapUnmappedMultiActions = function _mapUnmappedMultiActions() {
        var unmappedMultiActions = [];
        $('.action_buttons .extended_multi_actions a,' + '.action_buttons .extended_multi_actions button,' + '.action_buttons .extended_multi_actions input[type="button"],' + '.action_buttons .extended_multi_actions input[type="submit"]').each(function () {
            if (!_alreadyMapped($(this))) {
                unmappedMultiActions.push($(this));
            }
        });

        var $dropdown = $('#orders-table-dropdown');
        $.each(unmappedMultiActions, function () {
            var $button = $(this);
            var callback = function callback() {
                $button.get(0).click();
            };

            jse.libs.button_dropdown.mapAction($dropdown, $button.text(), '', callback);
            mappedButtons.push($button);
        });
    };

    /**
     * Checks if the button was already mapped
     *
     * @private
     */
    var _alreadyMapped = function _alreadyMapped($button) {
        for (var index in mappedButtons) {
            if ($button.is(mappedButtons[index])) {
                return true;
            }
        }
        return false;
    };

    /**
     * Add Button to Mapped Array
     *
     * @param buttonSelector
     * @returns {boolean}
     *
     * @private
     */
    var _addButtonToMappedArray = function _addButtonToMappedArray(buttonSelector) {
        if (mappedButtons[buttonSelector] !== undefined) {
            return true;
        }
        mappedButtons[buttonSelector] = $(buttonSelector);
    };

    /**
     * Bind Event handler
     *
     * @param $dropdown
     * @param action
     * @param customRecentButtonSelector
     *
     * @private
     */
    var _bindEventHandler = function _bindEventHandler($dropdown, action, customRecentButtonSelector) {
        var targetSelector = _selectorMapping[action],
            section = _sectionMapping[action],
            callback = _getActionCallback(action),
            customElement = $(customRecentButtonSelector).length ? $(customRecentButtonSelector) : $dropdown;
        if ($(targetSelector).length) {
            _addButtonToMappedArray(targetSelector);
            jse.libs.button_dropdown.mapAction($dropdown, action, section, callback, customElement);
        }
    };

    /**
     * Fix for row selection controls.
     *
     * @private
     */
    var _fixRowSelectionForControlElements = function _fixRowSelectionForControlElements() {
        $('input.checkbox[name="gm_multi_status[]"]').add('.single-checkbox').add('a.action-icon').add('.js-button-dropdown').add('tr.dataTableRow a').on('click', function (event) {
            event.stopPropagation();
            _toggleMultiActionButton();
        });
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Wait until the buttons are converted to dropdown for every row.
        var interval = setInterval(function () {
            if ($('.js-button-dropdown').length) {
                clearInterval(interval);

                _mapTableActions();
                _mapUnmappedMultiActions();

                var tableActions = mappedButtons;

                // Remove Mailbeez conversations badge.
                _addButtonToMappedArray('.contentTable .infoBoxContent a.context_view_button.btn_right');

                $('.gx-orders-table tr').not('.dataTableHeadingRow').each(function () {
                    mappedButtons = [];

                    for (var index in tableActions) {
                        mappedButtons[index] = tableActions[index];
                    }

                    _mapRowAction($(this));
                    _mapUnmappedRowActions($(this));
                });

                _fixRowSelectionForControlElements();

                // Initialize checkboxes
                _toggleMultiActionButton();
            }
        }, 300);

        // Check for selected checkboxes also
        // before all rows and their dropdown widgets have been initialized.
        _toggleMultiActionButton();

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
