<?php
/* --------------------------------------------------------------
  ProductListingPriceVariantsHandlerHelper.php 2023-06-13
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2023 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\Modules\ProductListing\App\Data\Price\Helper;

use Gambio\Shop\Modules\ProductListing\App\Data\Price\Components\ProductListingPriceCalculationHelper;
use Gambio\Shop\Modules\ProductListing\App\Data\Price\Components\ProductListingPriceFactory;
use Gambio\Shop\Modules\ProductListing\App\Data\Price\Components\ProductListingPriceSpecialRepository;
use Gambio\Shop\Modules\ProductListing\App\Data\Price\Components\ProductListingPriceTextFormatter;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\InternalToBeRenamed\ListingItemCurrency;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\InternalToBeRenamed\ListingItemPriceInformation;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\ListingItemId;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\ListingItemPriceSpecial;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\ListingItemPriceValue;
use Gambio\Shop\Modules\ProductListing\Model\ValueObjects\ListingSettings;

/**
 * Class ProductListingPriceVariantsHandlerHelper
 *
 * @package Gambio\Shop\Modules\ProductListing\App\Data\Price\Helper
 */
class ProductListingPriceVariantsHandlerHelper
{
    private ProductListingPriceCalculationHelper $calculationHelper;
    private ProductListingPriceFactory           $factory;
    private ProductListingPriceTextFormatter     $formatter;
    private ProductListingPriceSpecialRepository $priceSpecialRepository;
    
    
    /**
     * @param ProductListingPriceCalculationHelper $calculationHelper
     * @param ProductListingPriceFactory           $factory
     * @param ProductListingPriceTextFormatter     $formatter
     * @param ProductListingPriceSpecialRepository $priceSpecialRepository
     */
    public function __construct(
        ProductListingPriceCalculationHelper $calculationHelper,
        ProductListingPriceFactory           $factory,
        ProductListingPriceTextFormatter     $formatter,
        ProductListingPriceSpecialRepository $priceSpecialRepository
    )
    {
        $this->calculationHelper      = $calculationHelper;
        $this->factory                = $factory;
        $this->formatter              = $formatter;
        $this->priceSpecialRepository = $priceSpecialRepository;
    }


    /**
     * Processes the final special offers price value.
     *
     * @param float                       $specialPrice
     * @param ListingItemPriceInformation $priceInformation
     * @param ListingItemCurrency         $currency
     *
     * @return ListingItemPriceValue
     */
    public function getPrice(
        float                       $specialPrice,
        ListingItemPriceInformation $priceInformation,
        ListingItemCurrency         $currency
    ): ListingItemPriceValue
    {
        $priceValue = $this->factory->createPriceValue($specialPrice);

        return $this->calculationHelper->addTaxAndApplyCurrency($priceValue, $priceInformation->taxRate(), $currency);
    }


    /**
     * Returns the formatted price value string for the cheapest variant.
     *
     * @param ListingItemPriceValue $priceValue
     * @param ListingSettings       $listingSettings
     * @param ListingItemCurrency   $currency
     *
     * @return string
     */
    public function getFormattedPriceValue(
        ListingItemPriceValue $priceValue,
        ListingSettings       $listingSettings,
        ListingItemCurrency   $currency
    ): string
    {
        return $this->formatter->variant($priceValue, $currency, $listingSettings);
    }


    /**
     * Creates a listing item special instance.
     *
     * @param ListingItemId               $itemId
     * @param ListingSettings             $listingSettings
     * @param ListingItemPriceInformation $priceInformation
     * @param ListingItemCurrency         $currency
     *
     * @return ListingItemPriceSpecial
     */
    public function getPriceSpecial(
        ListingItemId               $itemId,
        ListingSettings             $listingSettings,
        ListingItemPriceInformation $priceInformation,
        ListingItemCurrency         $currency
    ): ListingItemPriceSpecial
    {
        $specialPrice   = $this->priceSpecialRepository->findSpecialPrice($itemId);
        $basePriceValue = $this->factory->createPriceValue($specialPrice);
        $basePriceValue = $this->calculationHelper->addCheapestVariantTaxAndApplyCurrency(
            $itemId,
            $basePriceValue,
            $priceInformation->taxRate(),
            $currency
        );
        $baseFormatted  = $this->formatter->variantWithSpecial($basePriceValue, $currency, $listingSettings);

        return $this->factory->createSpecial($specialPrice, $baseFormatted);
    }
}