<?php
/* --------------------------------------------------------------
   CashFlowTechDocumentsController.inc.php 2019-01-30
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

class CashFlowTechDocumentsController extends HttpViewController
{
    const DOWNLOAD_INTERVAL = 1200;
    
    protected $hubAuthenticated = false;
    
    public function proceed(HttpContextInterface $httpContext)
    {
        $serverData = $this->httpContextReader->getServerData($httpContext);
        $clientKey = (string)gm_get_conf('GAMBIO_HUB_CLIENT_KEY');
        $this->hubAuthenticated = !empty($clientKey) &&
                                  !empty($serverData['HTTP_X_CLIENT_KEY']) &&
                                  $clientKey === $serverData['HTTP_X_CLIENT_KEY'];
        parent::proceed($httpContext);
    }
    
    protected function makePermissionDeniedResponse()
    {
        $responseData = [
            'error' => [
                'message' => 'authentication failed',
            ],
        ];
        $headers = [
            'HTTP/1.1 403 Permission denied',
        ];
        $response = new JsonHttpControllerResponse($responseData, $headers);
        return $response;
    }
    
    public function actionDefault()
    {
        return parent::actionDefault(); // TODO: Change the autogenerated stub
    }
    
    public function actionListInvoices()
    {
        if (!$this->hubAuthenticated) {
            return $this->makePermissionDeniedResponse();
        }
        $orderId = (int)$this->_getQueryParameter('orderId');
        $now = time();
        $downloadExpires = $now + self::DOWNLOAD_INTERVAL;
        /** @var InvoiceArchiveReadService $invoiceReader */
        $invoiceReader   = StaticGXCoreLoader::getService('InvoiceArchiveRead');
        $invoiceCollection = $invoiceReader->getInvoiceListByConditions(['orders_id' => $orderId]);
        $invoices = [];
        /** @var \InvoiceListItem $invoiceListItem */
        foreach ($invoiceCollection as $invoiceListItem) {
            $invoiceFile = $invoiceListItem->getInvoiceFilename();
            $fileName = $invoiceFile . '_' .
                        $downloadExpires . '_' .
                        hash('sha256', $downloadExpires . $invoiceFile . LogControl::get_secure_token());
            $invoices[] = [
                'invoiceId'     => $invoiceListItem->getInvoiceId(),
                'invoiceNumber' => $invoiceListItem->getInvoiceNumber(),
                'fileName'      => $fileName,
                'invoiceDate'   => $invoiceListItem->getInvoiceDate()->format('c'),
                'invoiceAmount' => $invoiceListItem->getTotalSum(),
            ];
        }
        
        $responseData = [
            'timeStamp' => $now,
            'orderId'   => $orderId,
            'invoices'  => $invoices,
        ];
        
        $headers = ['Content-Type: application/json'];
        return new JsonHttpControllerResponse($responseData, $headers);
    }
    
    public function actionDownloadInvoice()
    {
        $fileName      = $this->_getQueryParameter('file');
        $fileNameParts = [];
        if (preg_match('/^(\d+_[a-f0-9]{32}\.pdf)_(\d+)_([a-f0-9]{64})$/', $fileName, $fileNameParts) === false) {
            $headers = ['HTTP/1.1 403 Permission denied', 'Content-Type: application/json'];
            return new JsonHttpControllerResponse(['error' => ['message' => 'filename invalid']], $headers);
        }
    
        $invoiceFileName     = $fileNameParts[1];
        $expirationTimestamp = (int)$fileNameParts[2];
        $inputHash           = $fileNameParts[3];
        $controlHash         = hash('sha256', $expirationTimestamp . $invoiceFileName . LogControl::get_secure_token());
        if ($inputHash !== $controlHash) {
            $headers = ['HTTP/1.1 403 Permission denied', 'Content-Type: application/json'];
            return new JsonHttpControllerResponse(['error' => ['message' => 'hash invalid']], $headers);
        }
        if ($expirationTimestamp < time()) {
            $headers = ['HTTP/1.1 403 Permission denied', 'Content-Type: application/json'];
            return new JsonHttpControllerResponse(['error' => ['message' => 'download expired']], $headers);
        }
        
        $fileContent = file_get_contents(DIR_FS_CATALOG . 'export/invoice/' . $invoiceFileName);
        $headers = [
            'Content-Type: application/pdf',
            'Content-Disposition: inline; filename="' . $invoiceFileName . '"',
        ];
        return new HttpControllerResponse($fileContent, $headers);
    }
}
